package net.sf.saxon;

import java.io.File;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.List;

import javax.xml.transform.Source;
import javax.xml.transform.TransformerException;
import javax.xml.transform.sax.SAXSource;
import javax.xml.transform.stream.StreamSource;

import org.xml.sax.InputSource;
import org.xml.sax.XMLReader;

/**
 * This <B>Transform</B> class is the entry point to the Saxon XSLT Processor.
 * This class is provided to control the processor from the command line.
 * <p>
 * <p/>
 * The XSLT syntax supported conforms to the W3C XSLT 1.0 and XPath 1.0
 * recommendation. Only the transformation language is implemented (not the
 * formatting objects). Saxon extensions are documented in the file
 * extensions.html
 *
 * @author Michael H. Kay
 */

public class Transform {

    /**
     * Load a document, or all the documents in a directory, given a filename or URL
     * 
     * @param sourceFileName the name of the source file or directory
     * @param useURLs        true if the filename argument is to be treated as a URI
     * @param config         the Saxon configuration
     * @param useSAXSource   true if the method should use a SAXSource rather than a
     *                       StreamSource
     * @return if sourceFileName represents a single source document, return a
     *         Source object representing that document. If sourceFileName
     *         represents a directory, return a List containing multiple Source
     *         objects, one for each file in the directory.
     */

    public static Object loadDocuments(String sourceFileName, boolean useURLs, Configuration config,
	    boolean useSAXSource) throws TransformerException {

	Source sourceInput;
	XMLReader parser;
	if (useURLs || sourceFileName.startsWith("http:") || sourceFileName.startsWith("file:")) {
	    sourceInput = config.getURIResolver().resolve(sourceFileName, null);
	    if (sourceInput == null) {
		sourceInput = config.getSystemURIResolver().resolve(sourceFileName, null);
	    }
	    return sourceInput;
	} else if (sourceFileName.equals("-")) {
	    // take input from stdin
	    if (useSAXSource) {
		parser = config.getSourceParser();
		sourceInput = new SAXSource(parser, new InputSource(System.in));
	    } else {
		sourceInput = new StreamSource(System.in);
	    }
	    return sourceInput;
	} else {
	    File sourceFile = new File(sourceFileName);
	    if (!sourceFile.exists()) {
		throw new TransformerException("Source file " + sourceFile + " does not exist");
	    }
	    if (sourceFile.isDirectory()) {
		parser = config.getSourceParser();
		List result = new ArrayList(20);
		String[] files = sourceFile.list();
		for (int f = 0; f < files.length; f++) {
		    File file = new File(sourceFile, files[f]);
		    if (!file.isDirectory()) {
			if (useSAXSource) {
			    InputSource eis = new InputSource(file.toURI().toString());
			    sourceInput = new SAXSource(parser, eis);
			    // it's safe to use the same parser for each document, as they
			    // will be processed one at a time.
			} else {
			    sourceInput = new StreamSource(file.toURI().toString());
			}
			result.add(sourceInput);
		    }
		}
		return result;
	    } else {
		if (useSAXSource) {
		    InputSource eis = new InputSource(sourceFile.toURI().toString());
		    sourceInput = new SAXSource(config.getSourceParser(), eis);
		} else {
		    sourceInput = new StreamSource(sourceFile.toURI().toString());
		}
		return sourceInput;
	    }
	}
    }

    private static String getLocalFileName(Source source) {
	try {
	    String path = new URI(source.getSystemId()).getPath();
	    while (true) {
		int sep = path.indexOf('/');
		if (sep < 0) {
		    return path;
		} else {
		    path = path.substring(sep + 1);
		}
	    }
	} catch (URISyntaxException err) {
	    throw new IllegalArgumentException(err.getMessage());
	}
    }

}

//
// The contents of this file are subject to the Mozilla Public License Version 1.0 (the "License");
// you may not use this file except in compliance with the License. You may obtain a copy of the
// License at http://www.mozilla.org/MPL/
//
// Software distributed under the License is distributed on an "AS IS" basis,
// WITHOUT WARRANTY OF ANY KIND, either express or implied.
// See the License for the specific language governing rights and limitations under the License.
//
// The Original Code is: all this file.
//
// The Initial Developer of the Original Code is Michael H. Kay.
//
// Portions created by (your name) are Copyright (C) (your legal entity). All Rights Reserved.
//
// Contributor(s): changes to allow source and/or stylesheet from stdin contributed by
// Gunther Schadow [gunther@aurora.regenstrief.org]
//arnaud.mergey@semarchy.com
