package net.sf.saxon.style;

import java.net.URI;
import java.net.URISyntaxException;

import net.sf.saxon.expr.Atomizer;
import net.sf.saxon.expr.ContextItemExpression;
import net.sf.saxon.expr.Expression;
import net.sf.saxon.expr.ExpressionTool;
import net.sf.saxon.expr.Literal;
import net.sf.saxon.expr.RoleLocator;
import net.sf.saxon.expr.StaticContext;
import net.sf.saxon.expr.StringLiteral;
import net.sf.saxon.expr.TypeChecker;
import net.sf.saxon.instruct.Executable;
import net.sf.saxon.om.AttributeCollection;
import net.sf.saxon.om.Axis;
import net.sf.saxon.om.NamespaceConstant;
import net.sf.saxon.om.StandardNames;
import net.sf.saxon.sort.CodepointCollator;
import net.sf.saxon.sort.SortKeyDefinition;
import net.sf.saxon.sort.StringCollator;
import net.sf.saxon.trans.XPathException;
import net.sf.saxon.type.ItemType;
import net.sf.saxon.value.EmptySequence;
import net.sf.saxon.value.SequenceType;
import net.sf.saxon.value.StringValue;
import net.sf.saxon.value.Whitespace;

/**
 * An xsl:sort element in the stylesheet. <br>
 */

public class XSLSort extends StyleElement {

    private SortKeyDefinition sortKeyDefinition;
    private Expression select;
    private Expression order;
    private Expression dataType = null;
    private Expression caseOrder;
    private Expression lang;
    private Expression collationName;
    private Expression stable;
    private boolean useDefaultCollation = true;

    /**
     * Determine whether this type of element is allowed to contain a sequence
     * constructor
     * 
     * @return true: yes, it may contain a sequence constructor
     */

    @Override
    public boolean mayContainSequenceConstructor() {
	return true;
    }

    @Override
    public void prepareAttributes() throws XPathException {

	AttributeCollection atts = getAttributeList();

	String selectAtt = null;
	String orderAtt = null;
	String dataTypeAtt = null;
	String caseOrderAtt = null;
	String langAtt = null;
	String collationAtt = null;
	String stableAtt = null;

	for (int a = 0; a < atts.getLength(); a++) {
	    int nc = atts.getNameCode(a);
	    String f = getNamePool().getClarkName(nc);
	    switch (f) {
	    case StandardNames.SELECT -> selectAtt = atts.getValue(a);
	    case StandardNames.ORDER -> orderAtt = Whitespace.trim(atts.getValue(a));
	    case StandardNames.DATA_TYPE -> dataTypeAtt = Whitespace.trim(atts.getValue(a));
	    case StandardNames.CASE_ORDER -> caseOrderAtt = Whitespace.trim(atts.getValue(a));
	    case StandardNames.LANG -> langAtt = Whitespace.trim(atts.getValue(a));
	    case StandardNames.COLLATION -> collationAtt = Whitespace.trim(atts.getValue(a));
	    case StandardNames.STABLE -> stableAtt = Whitespace.trim(atts.getValue(a));
	    default -> checkUnknownAttribute(nc);
	    }
	}

	if (selectAtt != null) {
	    select = makeExpression(selectAtt);
	}

	if (orderAtt == null) {
	    order = new StringLiteral("ascending");
	} else {
	    order = makeAttributeValueTemplate(orderAtt);
	}

	if (dataTypeAtt == null) {
	    dataType = null;
	} else {
	    dataType = makeAttributeValueTemplate(dataTypeAtt);
	}

	if (caseOrderAtt == null) {
	    caseOrder = new StringLiteral("#default");
	} else {
	    caseOrder = makeAttributeValueTemplate(caseOrderAtt);
	    useDefaultCollation = false;
	}

	if (langAtt == null) {
	    lang = new StringLiteral(StringValue.EMPTY_STRING);
	} else if (langAtt.equals("")) {
	    compileError("The lang attribute must be a valid language code", "XTDE0030");
	} else {
	    lang = makeAttributeValueTemplate(langAtt);
	    useDefaultCollation = false;
	}

	if (stableAtt == null) {
	    stable = null;
	} else {
	    stable = makeAttributeValueTemplate(stableAtt);
	}

	if (collationAtt != null) {
	    collationName = makeAttributeValueTemplate(collationAtt);
	    useDefaultCollation = false;
	}

    }

    @Override
    public void validate() throws XPathException {
	if (select != null && hasChildNodes()) {
	    compileError("An xsl:sort element with a select attribute must be empty", "XTSE1015");
	}
	if (select == null && !hasChildNodes()) {
	    select = new ContextItemExpression();
	}

	// Get the named or default collation

	if (useDefaultCollation) {
	    collationName = new StringLiteral(getDefaultCollationName());
	}

	StringCollator stringCollator = null;
	if (collationName instanceof StringLiteral) {
	    String collationString = ((StringLiteral) collationName).getStringValue();
	    try {
		URI collationURI = new URI(collationString);
		if (!collationURI.isAbsolute()) {
		    URI base = new URI(getBaseURI());
		    collationURI = base.resolve(collationURI);
		    collationString = collationURI.toString();
		}
	    } catch (URISyntaxException err) {
		compileError("Collation name '" + collationString + "' is not a valid URI");
		collationString = NamespaceConstant.CODEPOINT_COLLATION_URI;
	    }
	    stringCollator = getPrincipalStylesheet().findCollation(collationString);
	    if (stringCollator == null) {
		compileError("Collation " + collationString + " has not been defined", "XTDE1035");
		stringCollator = CodepointCollator.getInstance(); // for recovery paths
	    }
	}

	select = typeCheck("select", select);
	order = typeCheck("order", order);
	caseOrder = typeCheck("case-order", caseOrder);
	lang = typeCheck("lang", lang);
	dataType = typeCheck("data-type", dataType);
	collationName = typeCheck("collation", collationName);
	stable = typeCheck("stable", stable);

	if (select != null) {
	    try {
		RoleLocator role = new RoleLocator(RoleLocator.INSTRUCTION, "xsl:sort/select", 0);
		select = TypeChecker.staticTypeCheck(select, SequenceType.ATOMIC_SEQUENCE, false, role,
			makeExpressionVisitor());
	    } catch (XPathException err) {
		compileError(err);
	    }
	}

	sortKeyDefinition = new SortKeyDefinition();
	sortKeyDefinition.setOrder(order);
	sortKeyDefinition.setCaseOrder(caseOrder);
	sortKeyDefinition.setLanguage(lang);
	sortKeyDefinition.setSortKey(select);
	sortKeyDefinition.setDataTypeExpression(dataType);
	sortKeyDefinition.setCollationNameExpression(collationName);
	sortKeyDefinition.setCollation(stringCollator);
	sortKeyDefinition.setBaseURI(getBaseURI());
	sortKeyDefinition.setStable(stable);
	sortKeyDefinition.setBackwardsCompatible(backwardsCompatibleModeIsEnabled());
    }

    /**
     * Determine the type of item returned by this instruction (only relevant if it
     * is an instruction). Default implementation returns Type.ITEM, indicating that
     * we don't know, it might be anything. Returns null in the case of an element
     * such as xsl:sort or xsl:variable that can appear in a sequence constructor
     * but contributes nothing to the result sequence.
     * 
     * @return the item type returned
     */

    @Override
    protected ItemType getReturnedItemType() {
	return null;
    }

    @Override
    public Expression compile(Executable exec) throws XPathException {
	if (select == null) {
	    Expression b = compileSequenceConstructor(exec, iterateAxis(Axis.CHILD), true);
	    b.setContainer(this);

	    try {
		StaticContext env = getStaticContext();
		Atomizer atomizedSortKey = new Atomizer(makeExpressionVisitor().simplify(b), env.getConfiguration());
		ExpressionTool.copyLocationInfo(b, atomizedSortKey);
		sortKeyDefinition.setSortKey(atomizedSortKey);
	    } catch (XPathException e) {
		compileError(e);
	    }
	}
	// Simplify the sort key definition - this is especially important in the case
	// where
	// all aspects of the sort key are known statically.
	sortKeyDefinition = sortKeyDefinition.simplify(makeExpressionVisitor());
	// not an executable instruction
	return null;
    }

    public SortKeyDefinition getSortKeyDefinition() {
	return sortKeyDefinition;
    }

    public Expression getStable() {
	return stable;
    }

}

//
// The contents of this file are subject to the Mozilla Public License Version 1.0 (the "License");
// you may not use this file except in compliance with the License. You may obtain a copy of the
// License at http://www.mozilla.org/MPL/
//
// Software distributed under the License is distributed on an "AS IS" basis,
// WITHOUT WARRANTY OF ANY KIND, either express or implied.
// See the License for the specific language governing rights and limitations under the License.
//
// The Original Code is: all this file.
//
// The Initial Developer of the Original Code is Michael H. Kay.
//
// Portions created by (your name) are Copyright (C) (your legal entity). All Rights Reserved.
//
// Contributor(s): none.
//arnaud.mergey@emarchy.com
