package net.sf.saxon.value;

import net.sf.saxon.om.FastStringBuffer;

import java.math.BigInteger;

/**
 * This is a utility class that handles formatting of numbers as strings.
 * <p>
 * The algorithm for converting a floating point number to a string is taken from Guy L. Steele and
 * Jon L. White, <i>How to Print Floating-Point Numbers Accurately</i>, ACM SIGPLAN 1990. It is algorithm
 * (FPP)<sup>2</sup> from that paper. There are three separate implementations of the algorithm:
 * <ul>
 * <li>One using long arithmetic and generating non-exponential output representations
 * <li>One using BigInteger arithmetic and generating non-exponential output representation
 * <li>One using BigInteger arithmetic and generating exponential output representations
 * </ul>
 * <p>
 * The choice of method depends on the value of the number being formatted.
 * <p>
 * The module contains some residual code (mainly the routine for formatting integers) from the class
 * AppenderHelper by Jack Shirazi in the O'Reilly book <i>Java Performance Tuning</i>. The floating point routines
 * in that module were found to be unsuitable, since they used floating point arithmetic which introduces
 * rounding errors.
 * <p>
 * There are several reasons for doing this conversion within Saxon, rather than leaving it all to Java.
 * Firstly, there are differences in the required output format, notably the absence of ".0" when formatting
 * whole numbers, and the different rules for the range of numbers where exponential notation is used.
 * Secondly, there are bugs in some Java implementations, for example JDK outputs 0.001 as 0.0010, and
 * IKVM/GNU gets things very wrong sometimes. Finally, this implementation is faster for "everyday" numbers,
 * though it is slower for more extreme numbers. It would probably be reasonable to hand over formatting
 * to the Java platform (at least when running the Sun JDK) for exponents outside the range -7 to +7.
 */

public class FloatingPointConverter {

    public static FloatingPointConverter THE_INSTANCE = new FloatingPointConverter();

    private FloatingPointConverter(){}

    /**
     * char array holding the characters for the string "-Infinity".
     */
    private static final char[] NEGATIVE_INFINITY = {'-', 'I', 'N', 'F'};
    /**
     * char array holding the characters for the string "Infinity".
     */
    private static final char[] POSITIVE_INFINITY = {'I', 'N', 'F'};
    /**
     * char array holding the characters for the string "NaN".
     */
    private static final char[] NaN = {'N', 'a', 'N'};

    private static final char[] charForDigit = {
        '0', '1', '2', '3', '4', '5', '6', '7', '8', '9'
    };

    private static final long doubleSignMask = 0x8000000000000000L;
    private static final long doubleExpMask = 0x7ff0000000000000L;
    private static final int doubleExpShift = 52;
    private static final int doubleExpBias = 1023;
    private static final long doubleFractMask = 0xfffffffffffffL;
    private static final int floatSignMask = 0x80000000;
    private static final int floatExpMask = 0x7f800000;
    private static final int floatExpShift = 23;
    private static final int floatExpBias = 127;
    private static final int floatFractMask = 0x7fffff;

    private static final BigInteger TEN = BigInteger.valueOf(10);
    private static final BigInteger NINE = BigInteger.valueOf(9);

    /**
     * Format an integer, appending the string representation of the integer to a string buffer
     * @param s the string buffer
     * @param i the integer to be formatted
     * @return the supplied string buffer, containing the appended integer
     */

    public static FastStringBuffer appendInt(FastStringBuffer s, int i) {
        if (i < 0) {
            if (i == Integer.MIN_VALUE) {
                //cannot make this positive due to integer overflow
                s.append("-2147483648");
                return s;
            }
            s.append('-');
            i = -i;
        }
        int c;
        if (i < 10) {
            //one digit
            s.append(charForDigit[i]);
            return s;
        } else if (i < 100) {
            //two digits
            s.append(charForDigit[i / 10]);
            s.append(charForDigit[i % 10]);
            return s;
        } else if (i < 1000) {
            //three digits
            s.append(charForDigit[i / 100]);
            s.append(charForDigit[(c = i % 100) / 10]);
            s.append(charForDigit[c % 10]);
            return s;
        } else if (i < 10000) {
            //four digits
            s.append(charForDigit[i / 1000]);
            s.append(charForDigit[(c = i % 1000) / 100]);
            s.append(charForDigit[(c %= 100) / 10]);
            s.append(charForDigit[c % 10]);
            return s;
        } else if (i < 100000) {
            //five digits
            s.append(charForDigit[i / 10000]);
            s.append(charForDigit[(c = i % 10000) / 1000]);
            s.append(charForDigit[(c %= 1000) / 100]);
            s.append(charForDigit[(c %= 100) / 10]);
            s.append(charForDigit[c % 10]);
            return s;
        } else if (i < 1000000) {
            //six digits
            s.append(charForDigit[i / 100000]);
            s.append(charForDigit[(c = i % 100000) / 10000]);
            s.append(charForDigit[(c %= 10000) / 1000]);
            s.append(charForDigit[(c %= 1000) / 100]);
            s.append(charForDigit[(c %= 100) / 10]);
            s.append(charForDigit[c % 10]);
            return s;
        } else if (i < 10000000) {
            //seven digits
            s.append(charForDigit[i / 1000000]);
            s.append(charForDigit[(c = i % 1000000) / 100000]);
            s.append(charForDigit[(c %= 100000) / 10000]);
            s.append(charForDigit[(c %= 10000) / 1000]);
            s.append(charForDigit[(c %= 1000) / 100]);
            s.append(charForDigit[(c %= 100) / 10]);
            s.append(charForDigit[c % 10]);
            return s;
        } else if (i < 100000000) {
            //eight digits
            s.append(charForDigit[i / 10000000]);
            s.append(charForDigit[(c = i % 10000000) / 1000000]);
            s.append(charForDigit[(c %= 1000000) / 100000]);
            s.append(charForDigit[(c %= 100000) / 10000]);
            s.append(charForDigit[(c %= 10000) / 1000]);
            s.append(charForDigit[(c %= 1000) / 100]);
            s.append(charForDigit[(c %= 100) / 10]);
            s.append(charForDigit[c % 10]);
            return s;
        } else if (i < 1000000000) {
            //nine digits
            s.append(charForDigit[i / 100000000]);
            s.append(charForDigit[(c = i % 100000000) / 10000000]);
            s.append(charForDigit[(c %= 10000000) / 1000000]);
            s.append(charForDigit[(c %= 1000000) / 100000]);
            s.append(charForDigit[(c %= 100000) / 10000]);
            s.append(charForDigit[(c %= 10000) / 1000]);
            s.append(charForDigit[(c %= 1000) / 100]);
            s.append(charForDigit[(c %= 100) / 10]);
            s.append(charForDigit[c % 10]);
            return s;
        } else {
            //ten digits
            s.append(charForDigit[i / 1000000000]);
            s.append(charForDigit[(c = i % 1000000000) / 100000000]);
            s.append(charForDigit[(c %= 100000000) / 10000000]);
            s.append(charForDigit[(c %= 10000000) / 1000000]);
            s.append(charForDigit[(c %= 1000000) / 100000]);
            s.append(charForDigit[(c %= 100000) / 10000]);
            s.append(charForDigit[(c %= 10000) / 1000]);
            s.append(charForDigit[(c %= 1000) / 100]);
            s.append(charForDigit[(c %= 100) / 10]);
            s.append(charForDigit[c % 10]);
            return s;
        }
    }

    /**
     * Implementation of the (FPP)2 algorithm from Steele and White, for doubles in the range
     * 0.01 to 1000000, and floats in the range 0.000001 to 1000000.
     * In this range (a) XPath requires that the output should not be in exponential
     * notation, and (b) the arithmetic can be handled using longs rather than BigIntegers
     * @param sb the string buffer to which the formatted result is to be appended
     * @param e the exponent of the floating point number
     * @param f the fraction part of the floating point number, such that the "real" value of the
     * number is f * 2^(e-p), with p>=0 and 0 lt f lt 2^p
     * @param p the precision
     */

    private static void fppfpp(FastStringBuffer sb, int e, long f, int p) {
        long r = f << Math.max(e-p, 0);
        long s = 1L << Math.max(0, -(e-p));
        long mminus = 1L << Math.max(e-p, 0);
        long mplus = mminus;
        boolean initial = true;

        // simpleFixup

        if (f == 1L << (p-1)) {
            mplus = mplus << 1;
            r = r << 1;
            s = s << 1;
        }
        int k = 0;
        while (r < (s+9)/10) {  // (S+9)/10 == ceiling(S/10)
            k--;
            r = r*10;
            mminus = mminus * 10;
            mplus = mplus * 10;
        }
        while (2*r + mplus >= 2*s) {
            s = s*10;
            k++;
        }

        for (int z=k; z<0; z++) {
            if (initial) {
                sb.append("0.");
            }
            initial = false;
            sb.append('0');
        }

        // end simpleFixup

        boolean low;
        boolean high;
        int u;
        while (true) {
            k--;
            long r10 = r*10;
            u = (int)(r10 / s);
            r = r10 - (u * s);    // = R*10 % S, but faster - saves a division
            mminus = mminus * 10;
            mplus = mplus * 10;
            low = 2*r < mminus;
            high = 2*r > 2*s - mplus;
            if (low || high) break;
            if (k == -1) {
                if (initial) {
                    sb.append('0');
                }
                sb.append('.');
            }
            sb.append(charForDigit[u]);
            initial = false;
        }
        if (high && (!low || 2*r > s)) {
            u++;
        }
        if (k == -1) {
            if (initial) {
                sb.append('0');
            }
            sb.append('.');
        }
        sb.append(charForDigit[u]);
        for (int z=0; z<k; z++) {
            sb.append('0');
        }
    }

    /**
     * Implementation of the (FPP)2 algorithm from Steele and White, for doubles in the range
     * 0.000001 to 0.01. In this range XPath requires that the output should not be in exponential
     * notation, but the scale factors are large enough to exceed the capacity of long arithmetic.
     * @param sb the string buffer to which the formatted result is to be appended
     * @param e the exponent of the floating point number
     * @param f the fraction part of the floating point number, such that the "real" value of the
     * number is f * 2^(e-p), with p>=0 and 0 lt f lt 2^p
     * @param p the precision
     */

    private static void fppfppBig(FastStringBuffer sb, int e, long f, int p) {
        BigInteger r = BigInteger.valueOf(f).shiftLeft(Math.max(e-p, 0));

        BigInteger s = BigInteger.ONE.shiftLeft(Math.max(0, -(e-p)));

        BigInteger mminus = BigInteger.ONE.shiftLeft(Math.max(e-p, 0));

        BigInteger mplus = mminus;

        boolean initial = true;

        // simpleFixup

        if (f == 1L << (p-1)) {
            mplus = mplus.shiftLeft(1);
            r = r.shiftLeft(1);
            s = s.shiftLeft(1);
        }
        int k = 0;
        while (r.compareTo(s.add(NINE).divide(TEN)) < 0) {  // (S+9)/10 == ceiling(S/10)
            k--;
            r = r.multiply(TEN);
            mminus = mminus.multiply(TEN);
            mplus = mplus.multiply(TEN);
        }
        while (r.shiftLeft(1).add(mplus).compareTo(s.shiftLeft(1)) >= 0) {
            s = s.multiply(TEN);
            k++;
        }

        for (int z=k; z<0; z++) {
            if (initial) {
                sb.append("0.");
            }
            initial = false;
            sb.append('0');
        }

        // end simpleFixup

        boolean low;
        boolean high;
        int u;
        while (true) {
            k--;
            BigInteger r10 = r.multiply(TEN);
            u = r10.divide(s).intValue();
            r = r10.mod(s);
            mminus = mminus.multiply(TEN);
            mplus = mplus.multiply(TEN);
            BigInteger r2 = r.shiftLeft(1);
            low = r2.compareTo(mminus) < 0;
            high = r2.compareTo(s.shiftLeft(1).subtract(mplus)) > 0;
            if (low || high) break;
            if (k == -1) {
                if (initial) {
                    sb.append('0');
                }
                sb.append('.');
            }
            sb.append(charForDigit[u]);
            initial = false;
        }
        if (high && (!low || r.shiftLeft(1).compareTo(s) > 0)) {
            u++;
        }
        if (k == -1) {
            if (initial) {
                sb.append('0');
            }
            sb.append('.');
        }
        sb.append(charForDigit[u]);
        for (int z=0; z<k; z++) {
            sb.append('0');
        }
    }


    /**
     * Implementation of the (FPP)2 algorithm from Steele and White, for numbers outside the range
     * 0.000001 to 1000000. In this range XPath requires that the output should be in exponential
     * notation
     * @param sb the string buffer to which the formatted result is to be appended
     * @param e the exponent of the floating point number
     * @param f the fraction part of the floating point number, such that the "real" value of the
     * number is f * 2^(e-p), with p>=0 and 0 lt f lt 2^p
     * @param p the precision
     */

    private static void fppfppExponential(FastStringBuffer sb, int e, long f, int p) {
        BigInteger r = BigInteger.valueOf(f).shiftLeft(Math.max(e-p, 0));

        BigInteger s = BigInteger.ONE.shiftLeft(Math.max(0, -(e-p)));

        BigInteger mminus = BigInteger.ONE.shiftLeft(Math.max(e-p, 0));

        BigInteger mplus = mminus;

        boolean initial = true;
        boolean doneDot = false;

        // simpleFixup

        if (f == 1L << (p-1)) {
            mplus = mplus.shiftLeft(1);
            r = r.shiftLeft(1);
            s = s.shiftLeft(1);
        }
        int k = 0;
        while (r.compareTo(s.add(NINE).divide(TEN)) < 0) {  // (S+9)/10 == ceiling(S/10)
            k--;
            r = r.multiply(TEN);
            mminus = mminus.multiply(TEN);
            mplus = mplus.multiply(TEN);
        }
        while (r.shiftLeft(1).add(mplus).compareTo(s.shiftLeft(1)) >= 0) {
            s = s.multiply(TEN);
            k++;
        }

        // end simpleFixup

        int h = k-1;

        boolean low;
        boolean high;
        int u;
        while (true) {
            k--;
            BigInteger r10 = r.multiply(TEN);
            u = r10.divide(s).intValue();
            r = r10.mod(s);
            mminus = mminus.multiply(TEN);
            mplus = mplus.multiply(TEN);
            BigInteger r2 = r.shiftLeft(1);
            low = r2.compareTo(mminus) < 0;
            high = r2.compareTo(s.shiftLeft(1).subtract(mplus)) > 0;
            if (low || high) break;

            sb.append(charForDigit[u]);
            if (initial) {
                sb.append('.');
                doneDot = true;
            }
            initial = false;
        }
        if (high && (!low || r.shiftLeft(1).compareTo(s) > 0)) {
            u++;
        }
        sb.append(charForDigit[u]);

        if (!doneDot) {
            sb.append(".0");
        }
        sb.append('E');
        appendInt(sb, h);

    }

    /**
     * Append a string representation of a double value to a string buffer
     * @param s the string buffer to which the result will be appended
     * @param value the double to be formatted
     * @return the original string buffer, now containing the string representation of the supplied double
     */

    public static FastStringBuffer appendDouble(FastStringBuffer s, double value) {
        double d = value;
        if (d == Double.NEGATIVE_INFINITY) {
            s.append(NEGATIVE_INFINITY);
        } else if (d == Double.POSITIVE_INFINITY) {
            s.append(POSITIVE_INFINITY);
        } else if (Double.isNaN(d)) {
            s.append(NaN);
        } else if (d == 0.0) {
            if ((Double.doubleToLongBits(d) & doubleSignMask) != 0) {
                s.append('-');
            }
            s.append('0');
        } else if (d == Double.MAX_VALUE) {
            s.append("1.7976931348623157E308");
        } else if (d == -Double.MAX_VALUE) {
            s.append("-1.7976931348623157E308");
        } else if (d == Double.MIN_VALUE) {
            s.append("4.9E-324");
        } else if (d == -Double.MIN_VALUE) {
            s.append("-4.9E-324");
        } else {
            if (d < 0) {
                s.append('-');
                d = -d;
            }
            boolean exponential = (d >= 1000000 || d < 0.000001);
            long bits = Double.doubleToLongBits(d);
            long fraction = (1L<<52) | (bits & doubleFractMask);
            long rawExp = (bits & doubleExpMask) >> doubleExpShift;
            int exp = (int)rawExp - doubleExpBias;
            if (rawExp == 0) {
                // don't know how to handle this currently: hand it over to Java to deal with
                s.append(Double.toString(value));
                return s;
            }
            if (exponential) {
                fppfppExponential(s, exp, fraction, 52);
            } else {
                if (d <= 0.01) {
                    fppfppBig(s, exp, fraction, 52);
                } else {
                    fppfpp(s, exp, fraction, 52);
                }
            }
        }
        return s;
    }

    /**
     * Append a string representation of a double value to a string buffer, forcing use of
     * exponential notation
     * @param s the string buffer to which the result will be appended
     * @param value the double to be formatted
     * @return the original string buffer, now containing the string representation of the supplied double
     */

    public static FastStringBuffer appendDoubleExponential(FastStringBuffer s, double value) {
        double d = value;
        if (d == Double.NEGATIVE_INFINITY) {
            s.append(NEGATIVE_INFINITY);
        } else if (d == Double.POSITIVE_INFINITY) {
            s.append(POSITIVE_INFINITY);
        } else if (Double.isNaN(d)) {
            s.append(NaN);
        } else if (d == 0.0) {
            if ((Double.doubleToLongBits(d) & doubleSignMask) != 0) {
                s.append('-');
            }
            s.append('0');
        } else if (d == Double.MAX_VALUE) {
            s.append("1.7976931348623157E308");
        } else if (d == -Double.MAX_VALUE) {
            s.append("-1.7976931348623157E308");
        } else if (d == Double.MIN_VALUE) {
            s.append("4.9E-324");
        } else if (d == -Double.MIN_VALUE) {
            s.append("-4.9E-324");
        } else {
            if (d < 0) {
                s.append('-');
                d = -d;
            }
            long bits = Double.doubleToLongBits(d);
            long fraction = (1L<<52) | (bits & doubleFractMask);
            long rawExp = (bits & doubleExpMask) >> doubleExpShift;
            int exp = (int)rawExp - doubleExpBias;
            if (rawExp == 0) {
                // don't know how to handle this currently: hand it over to Java to deal with
                s.append(Double.toString(value));
                return s;
            }
            fppfppExponential(s, exp, fraction, 52);
        }
        return s;
    }


    /**
     * Append a string representation of a float value to a string buffer
     * @param s the string buffer to which the result will be appended
     * @param value the float to be formatted
     * @return the original string buffer, now containing the string representation of the supplied float
     */

    public static FastStringBuffer appendFloat(FastStringBuffer s, float value) {
        float f = value;
        if (f == Float.NEGATIVE_INFINITY) {
            s.append(NEGATIVE_INFINITY);
        } else if (f == Float.POSITIVE_INFINITY) {
            s.append(POSITIVE_INFINITY);
        } else if (Double.isNaN(f)) {
            s.append(NaN);
        } else if (f == 0.0) {
            if ((Float.floatToIntBits(f) & floatSignMask) != 0) {
                s.append('-');
            }
            s.append('0');
        } else if (f == Float.MAX_VALUE) {
            s.append("3.4028235E38");
        } else if (f == -Float.MAX_VALUE) {
            s.append("-3.4028235E38");
        } else if (f == Float.MIN_VALUE) {
            s.append("1.4E-45");
        } else if (f == -Float.MIN_VALUE) {
            s.append("-1.4E-45");
        } else {
            if (f < 0) {
                s.append('-');
                f = -f;
            }
            boolean exponential = (f >= 1000000 || f < 0.000001F);
            int bits = Float.floatToIntBits(f);
            int fraction = (1<<23) | (bits & floatFractMask);
            int rawExp = ((bits & floatExpMask) >> floatExpShift);
            int exp = rawExp - floatExpBias;
            int precision = 23;
            if (rawExp == 0) {
                // don't know how to handle this currently: hand it over to Java to deal with
                s.append(Float.toString(value));
                return s;
            }
            if (exponential) {
                fppfppExponential(s, exp, fraction, precision);
            } else {
                fppfpp(s, exp, fraction, precision);
            }
        }
        return s;
    }

    /**
     * Append a string representation of a float value to a string buffer, forcing use of exponential
     * notation
     * @param s the string buffer to which the result will be appended
     * @param value the float to be formatted
     * @return the original string buffer, now containing the string representation of the supplied float
     */

    public static FastStringBuffer appendFloatExponential(FastStringBuffer s, float value) {
        float f = value;
        if (f == Float.NEGATIVE_INFINITY) {
            s.append(NEGATIVE_INFINITY);
        } else if (f == Float.POSITIVE_INFINITY) {
            s.append(POSITIVE_INFINITY);
        } else if (Double.isNaN(f)) {
            s.append(NaN);
        } else if (f == 0.0) {
            if ((Float.floatToIntBits(f) & floatSignMask) != 0) {
                s.append('-');
            }
            s.append('0');
        } else if (f == Float.MAX_VALUE) {
            s.append("3.4028235E38");
        } else if (f == -Float.MAX_VALUE) {
            s.append("-3.4028235E38");
        } else if (f == Float.MIN_VALUE) {
            s.append("1.4E-45");
        } else if (f == -Float.MIN_VALUE) {
            s.append("-1.4E-45");
        } else {
            if (f < 0) {
                s.append('-');
                f = -f;
            }
            int bits = Float.floatToIntBits(f);
            int fraction = (1<<23) | (bits & floatFractMask);
            int rawExp = ((bits & floatExpMask) >> floatExpShift);
            int exp = rawExp - floatExpBias;
            int precision = 23;
            if (rawExp == 0) {
                // don't know how to handle this currently: hand it over to Java to deal with
                s.append(Float.toString(value));
                return s;
            }
            fppfppExponential(s, exp, fraction, precision);
        }
        return s;
    }
}


//
// The contents of this file are subject to the Mozilla Public License Version 1.0 (the "License");
// you may not use this file except in compliance with the License. You may obtain a copy of the
// License at http://www.mozilla.org/MPL/
//
// Software distributed under the License is distributed on an "AS IS" basis,
// WITHOUT WARRANTY OF ANY KIND, either express or implied.
// See the License for the specific language governing rights and limitations under the License.
//
// The Original Code is: all this file.
//
// The Initial Developer of the Original Code is Michael H. Kay, based on a published algorithm by
// Guy L. Steele and Jon L. White.
//
// Contributor(s): the appendInt routine, and some of the constant declarations (and some of the ideas) are
// from the class AppenderHelper by Jack Shirazi in the O'Reilly book Java Performance Tuning..
// arnaud.mergey@semarchy.com
